
; WATER TANK LEVEL METER using pressure sensor
; REV1 to fix fault in the encode update where 4-7 was 2s rather than 33s and 8-B was 4s rather than 67s


	list P=16F88
	#include p16f88.inc
	ERRORLEVEL -302
	ERRORLEVEL -306

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_ALL & _CCP1_RB3 & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_ON & _MCLR_OFF & _PWRTE_ON & _WDT_ON & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

;EEPROM
EEPROM1		equ	H'00'	; non-volatile storage for temperature compensation flag
EEPROM2		equ	H'01'	; non-volatile storage for temperature compensation turnover ms
EEPROM3		equ	H'02'	; non-volatile storage for temperature compensation turnover ls

; Bank 0 RAM
TANK_No		equ	H'20'	; tank number
ENCODE_VAL	equ	H'21'	; encode value
TEMP_M		equ	H'22'	; temporary ms byte
TEMP_L		equ	H'23'	; temporary ls byte
SPAN_H_110	equ	H'24'	; ms byte of 110% span
SPAN_L_110	equ	H'25'	; ls byte of 110% span
COMPENSATE	equ	H'26'	; compensate flag (pressure vs temperature)
NEG			equ	H'27'	; negative value
TANK_LEV	equ	H'28'	; tank level in percentage
TEMP_C		equ	H'29'	; temperature in deg C
CELL_VOLT	equ	H'2A'	; cell voltage
KELVIN_0	equ	H'2B'	; ms byte of Kelvin temperature
KELVIN_1	equ	H'2C'	; ls byte of Kelvin temperature
MODULATE	equ	H'2D'	; modulation flags for LEDs
LED_PERIOD1	equ	H'2E'	; LED on period timer
LED_PERIOD2	equ	H'2F'	; LED on period timer	
CONVERSION	equ	H'30'	; conversion timer
STORE1		equ	H'31'	; timer counter
STORE2		equ	H'32'	; timer counter
TURNOVER_0	equ	H'33'	; turnover temperature ms
TURNOVER_1	equ	H'34'	; turnover temperature ls
TEMPER_M	equ	H'35'	; temporary ms byte
TEMPER_L	equ	H'36'	; temporary ls byte
TEMP_C1		equ	H'37'	; first calculation of temperature in deg C
SPAN_H		equ	H'38'	; ms byte of span
SPAN_L		equ	H'39'	; ls byte of span
MAX_PR_M	equ	H'3A'	; max pressure value ms byts
MAX_PR_L	equ	H'3B'	; max pressure value ls byte
TEMP_CAL	equ	H'3C'	; temperature turnover calibration flag

; Math
AARGB0		equ	H'50'	; MS Arguemnt
AARGB1		equ	H'51'	; 	
AARGB2		equ	H'52'	; 
AARGB3		equ	H'53'	; LS Argument
BARGB0		equ	H'54'	; MS Argument
BARGB1		equ	H'55'	; LS Argument
TEMPD		equ	H'56'	; temp
TEMPB0		equ	H'57'	; 
TEMPB1		equ	H'58'	; 
REMB0		equ	H'59'	; remainder
REMB1		equ	H'5A'	; 
LOOPCOUNT	equ	H'5B'	; counter 

; All Banks RAM
TEMP		equ	H'70'	; temporary register
COMP_FLG	equ	H'71'	; comparator flag

; preset EEPROM
	ORG     H'2100'
	DE		H'00'		; initially setting for temperature compensation (zero)
; other settings available are 1/7, 2/7, 3/7,... 8/7, 9/7 (ie 9 settings plus the 0 = 10)
	DE		H'01'		; initial temperature turnover (21 deg C)ms byte (set by 0V (short) at cell input)
	DE		H'26'		; initial temperature turnover ls byte

; start at memory 0
	org	0				; reset vector
	goto	SETUP
	org 4				; interrupt vector not used
	goto	SETUP

; osctune lookup table for tank number (0-9)
OSCTUNE_TANK
	addwf	PCL,f		; add value to program counter
	retlw	D'00'		; tank number 0
	retlw	D'03'		; tank number 1	
	retlw	D'06'		; tank number 2
	retlw	D'09'		; tank number 3
	retlw	D'12'		; tank number 4
	retlw	D'15'		; tank number 5
	retlw	D'18'		; tank number 6
	retlw	D'21'		; tank number 7
	retlw	D'24'		; tank number 8
	retlw	D'27'		; tank number 9

; osctune lookup table for encode number (0-15)
OSCTUNE_ENCODE
	addwf	PCL,f		; add value to program counter
	retlw	D'00'		; encode value 0
	retlw	D'01'		; encode value 1	
	retlw	D'02'		; encode value 2
	retlw	D'00'		; encode value 3
	retlw	D'01'		; encode value 4
	retlw	D'02'		; encode value 5
	retlw	D'00'		; encode value 6
	retlw	D'01'		; encode value 7
	retlw	D'02'		; encode value 8
	retlw	D'00'		; encode value 9
	retlw	D'01'		; encode value 10
	retlw	D'02'		; encode value 11	
	retlw	D'00'		; encode value 12
	retlw	D'01'		; encode value 13
	retlw	D'02'		; encode value 14
	retlw	D'00'		; encode value 15

; **********************************************************************************************
	
SETUP
	clrwdt				; watchdog cleared
	bcf		STATUS,RP0	; bank 0
	bcf		STATUS,RP1	
	clrf	COMP_FLG	; comparator flag indicator initially cleared

	bsf		STATUS,RP0	; select memory bank 1
	btfss	PCON,1		; bit clear if a power on reset
	goto	SET_COMP_FLAG
BROWN
	btfsc	PCON,0		; if set then no brownout
	goto	SET_INPUTS_OUTPUTS
SET_COMP_FLAG
	bsf		COMP_FLG,0	; set comparator flag indicator if power on reset or brownout

SET_INPUTS_OUTPUTS
; EEPROM values
	movlw	EEPROM1		; get compensate flag
	call	EEREAD
	movwf	COMPENSATE
	movlw	EEPROM2		; get turnover temperature
	call	EEREAD
	movwf	TURNOVER_0	; ms byte
	movlw	EEPROM3		; get turnover temperature
	call	EEREAD
	movwf	TURNOVER_1	; ls byte

	bsf		STATUS,RP0	; select memory bank 1
	bsf		PCON,1		; set so not cleared unless a power on reset
	bsf		PCON,0		; no brownout reset
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'01111111'	; port B outputs/ inputs set 
	movwf	TRISB		; port B data direction register
	movlw	B'00101110'	; outputs (0) and inputs (1)
	movwf	TRISA
; initially set to /128 for 2.097s x 128 or  4.47 minutes (altered with encode switch)
	movlw	B'00001111'	; reset at 00001111 settings (pullups enabled, WDT/128)
	movwf	OPTION_REG

; analog inputs, A/D

	movlw	B'00001110'	; AN1 to AN3 are analog inputs
	movwf	ANSEL
	movlw	B'11000000'	; right justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'01001001'	; Fosc, channel 1 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01111000'	; 0111 for 8MHz
	movwf	OSCCON		; 
	bsf		PIE2,CMIE	; comparator interrupt enable
	bsf		INTCON,PEIE	; peripheral interrupt enable
	bcf		STATUS,RP0	; select memory bank 0
	bsf		STATUS,RP1	; bank 2
	movlw	B'00010110'
	movwf   WDTCON		; set watchdog prescale at 65536 for 2.097s x (option register prescaler extension) 	
	bcf		STATUS,RP1	; select memory bank 0

; initialise ports
	bcf		PORTB,7		; data off
	movlw	B'00000000'	; RA0,RA6,RA7 low, RA4 low 
	movwf	PORTA		; RGB LED off, '5V switched' power on

; start up delay
POWER_ON
	movlw	D'255'		; approx 64ms 
	call	DELAYX		; delay
	movlw	D'255'		; approx 64ms 
	call	DELAYX		; delay

TANK_NUMBER
; read tank and encode values
; tank number (0-9)
	clrf	TANK_No		; tank zero
	btfss	PORTB,2		; if clear set bit 0	
	bsf		TANK_No,0
	btfss	PORTA,5		; if clear set bit 1	
	bsf		TANK_No,1
	btfss	PORTB,1		; if clear set bit 2	
	bsf		TANK_No,2
	btfss	PORTB,0		; if clear set bit 3	
	bsf		TANK_No,3

; Encode (0-15)
	clrf	ENCODE_VAL	; encode zero
	btfss	PORTB,3		; if clear set bit 0	
	bsf		ENCODE_VAL,0
	btfss	PORTB,5		; if clear set bit 1	
	bsf		ENCODE_VAL,1
	btfss	PORTB,6		; if clear set bit 2	
	bsf		ENCODE_VAL,2
	btfss	PORTB,4		; if clear set bit 3	
	bsf		ENCODE_VAL,3

; adjust osctune to alter 31.25kHz intrc oscillator rate (0.39% per value)
; so transmissions are at different rates for each tank	
; if tank =0 set at 0; 1, set at 3; 2, set at 6; 3 set at 9; 4 set at 12; 5, set at 15; 6, set at 18;
; 7, set at 21; 8, set at 24; 9, set at 27
	
	movf	TANK_No,w	; tank number
	call	OSCTUNE_TANK; get value to change osctune by
	movwf	TEMP		; temporary working register
	btfsc	TANK_No,0	; if odd, set ms bit in osctune (for reduction in frequency)
	bsf		TEMP,5		; reduce frequency bit

; if encode is 0,3,6,9,12,15 add 0
; if encode is 1,4,7,10,13 add 1
; if encode is 2,5,8,11,14 add 2

	movf	ENCODE_VAL,w; encode value
	call	OSCTUNE_ENCODE; get osctune increment value
	addwf	TEMP,w		; add to working register
	bsf		STATUS,RP0	; select memory bank 1
	movwf	OSCTUNE		; only different to previous value if switches are altered
	bcf		STATUS,RP0	; select memory bank 0

; alter update period with encode value
; encode 0,1,2,3 = 16.78s, 4,5,6,7 = 33.55s, 8,9,A,B = 67.1s, C,D,E,F 268.42s
	movf	ENCODE_VAL,w; encode value	
	movwf	TEMP		; all banks temp register
	sublw	D'03'		; compare with 3
	btfsc	STATUS,C
	goto	LESS_3		; less than 3
	movf	TEMP,w
	sublw	D'7'		; compare with 7
	btfsc	STATUS,C
	goto	LESS_7
	movf	TEMP,w
	sublw	H'B'		; compare with B
	btfsc	STATUS,C
	goto	LESS_B

; set at 268.42s
	bsf		STATUS,RP0	; select memory bank 1
	bsf		OPTION_REG,0
	bsf		OPTION_REG,1
	bsf		OPTION_REG,2; slowest rate
	bcf		STATUS,RP0	; select memory bank 0
	goto	CH_1AD
LESS_3
; set at 16,78s
	bsf		STATUS,RP0	; select memory bank 1
	bsf		OPTION_REG,0
	bsf		OPTION_REG,1
	bcf		OPTION_REG,2; rate
	bcf		STATUS,RP0	; select memory bank 0
	goto	CH_1AD
LESS_7
; set at 33.55s
	bsf		STATUS,RP0	; select memory bank 1
	bcf		OPTION_REG,0
	bcf		OPTION_REG,1
	bsf		OPTION_REG,2; rate
	bcf		STATUS,RP0	; select memory bank 0
	goto	CH_1AD
LESS_B
; set at 67.10s
	bsf		STATUS,RP0	; select memory bank 1
	bsf		OPTION_REG,0
	bcf		OPTION_REG,1
	bsf		OPTION_REG,2; rate
	bcf		STATUS,RP0	; select memory bank 0
	goto	CH_1AD	

; Channel A/D value 
CH_1AD ; cell voltage
	bsf		ADCON0,3
	bcf		ADCON0,4	; channel 1
	call	ACQUIRE_AD	; convert
	movf	ADRESH,w
	movwf	TEMP_M
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls bits
	bcf		STATUS,RP0	; bank 0
	movwf	TEMP_L

; if a low value set the turnover temperature calibration flag
	clrf	TEMP_CAL	; flag clear
	movf	TEMP_M,w
	btfss	STATUS,Z	; if ms byte not zero bypass	
	goto	SET_CH2		; not a low value

; ls bits
	movf	TEMP_L,w	; ls bits
	sublw	D'102'		; below 102? (0.5V)
	btfsc	STATUS,C
	bsf		TEMP_CAL,0	; set flag when level is low to indicate calibration is required

SET_CH2
; set A/D for CH2
	bcf		ADCON0,3
	bsf		ADCON0,4	; channel 2

; convert to voltage TEMP_H TEMP_L x 500V/1023
	movlw	H'1'		; H'1F4' is D500
	movwf	AARGB0
	movlw	H'F4'
	movwf	AARGB1
	movf	TEMP_M,w	; temp ms byte
	movwf	BARGB0
	movf	TEMP_L,w	; temp ls byte
	movwf	BARGB1
	call	FXM1616U	; multiply 

; divide by 1023
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
	movlw	H'3'		; D1023
	movwf	BARGB0
	movlw	H'FF'		; 
	movwf	BARGB1
	call	FXD2416U	; divide

; AARGB1 and AARGB2 has result 
	movf	AARGB1,w	; if not zero overrrange
	btfss	STATUS,Z
	clrf	AARGB2		; set at 0 if overrange
	btfsc	REMB0,7		; if remainder set increase value
	incf	AARGB2,f	; next	
	movf	AARGB2,w	; ls byte
	movwf	CELL_VOLT	; cell voltage

CH_2AD ; measure temperature
	bcf		ADCON0,3
	bsf		ADCON0,4	; channel 2
	call	ACQUIRE_AD	; convert
	movf	ADRESH,w
	movwf	TEMP_M		; temperature ms byte
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls bits
	bcf		STATUS,RP0	; bank 0
	movwf	TEMP_L

; check if a low value (less than 0.5V) if so alter temperature compensation
; this is tied to 0V at power up to allow the flag to change

	movf	TEMP_M,w
	btfss	STATUS,Z	; if ms byte not zero bypass	
	goto	MULT_500	; not a low value
; ls bits
	movf	TEMP_L,w	; ls bits
	sublw	D'102'		; below 102? (0.5V)
	btfss	STATUS,C
	goto	MULT_500
	incf	COMPENSATE,f; change value
	movf	COMPENSATE,w
	sublw	D'9'		; subtract from 9, if negative then set at 0
	btfss	STATUS,C
	clrf	COMPENSATE	; set at 0
	movlw	EEPROM1
	call	EEREAD		; sets EEADR
	movf	COMPENSATE,w
	call	EEWRITE
	
; drive RGB LED to show temperature compensation level
	movf	COMPENSATE,w
	xorlw	D'9'
	btfsc	STATUS,Z	; (White)
	goto	LED_9
	movf	COMPENSATE,w
	xorlw	D'8'
	btfsc	STATUS,Z	; (Violet)
	goto	LED_8
	movf	COMPENSATE,w
	xorlw	D'7'
	btfsc	STATUS,Z	; (Violet-Indigo)
	goto	LED_7
	movf	COMPENSATE,w
	xorlw	D'6'
	btfsc	STATUS,Z	; (Indigo)
	goto	LED_6
	movf	COMPENSATE,w
	xorlw	D'5'
	btfsc	STATUS,Z	; (Indigo-Blue)
	goto	LED_5
	movf	COMPENSATE,w
	xorlw	D'4'
	btfsc	STATUS,Z	; (blue)
	goto	LED_4
	movf	COMPENSATE,w
	xorlw	D'3'
	btfsc	STATUS,Z	; (Green)
	goto	LED_3
	movf	COMPENSATE,w
	xorlw	D'2'
	btfsc	STATUS,Z	; (Yellow)
	goto	LED_2
	movf	COMPENSATE,w
	xorlw	D'1'
	btfsc	STATUS,Z	; (Orange)
	goto	LED_1

LED_ZERO 
	clrf	MODULATE	; modulation of LED off 
	bsf		PORTA,7		; red LED
	bcf		PORTA,6		; green off
	bcf		PORTA,0		; blue off
	goto	LED_SHOW
LED_1 
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,7		; red LED on
	bsf		MODULATE,6	; modulate green LED at 50%
	bcf		PORTA,0		; blue off
	goto	LED_SHOW
LED_2 
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,7		; red LED on
	bsf		PORTA,6		; green LED on
	bcf		PORTA,0		; blue off
	goto	LED_SHOW
LED_3
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,6		; green LED on
	bcf		PORTA,0		; blue off
	bcf		PORTA,7		; red off
	goto	LED_SHOW
LED_4
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bcf		PORTA,6		; green off
	bcf		PORTA,7		; red off
	goto	LED_SHOW
LED_5
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		MODULATE,6	; modulate green LED at 50%	
	bcf		PORTA,7		; red off
	goto	LED_SHOW
LED_6
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		PORTA,6		; green LED on
	bcf		PORTA,7		; red off
	goto	LED_SHOW
LED_7
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		MODULATE,7	; modulate red LED at 50%
	bcf		PORTA,6		; green off
	goto	LED_SHOW
LED_8
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		PORTA,7		; red LED on
	bcf		PORTA,6		; green off
	goto	LED_SHOW
LED_9
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		PORTA,7		; red LED on
	bsf		PORTA,6		; green LED on

LED_SHOW
	movlw	D'85'
	movwf	LED_PERIOD1	; counters for LED lit period
LOOP_Z
	movlw	D'20'
	movwf	LED_PERIOD2	; counter for LED lit period
LOOP_C
	call	DELAY		; modulation period
; modulate LEDs
	btfsc	MODULATE,6	; if set switch green LED on and off
	call	MOD_GREEN
	btfsc	MODULATE,7
	call	MOD_RED
	decfsz	LED_PERIOD2,f; reduce to zero
	goto	LOOP_C
	decfsz	LED_PERIOD1,f; when zero end of LED display	
	goto	LOOP_Z
	goto	BYPASS_LED
	
; temperature calibration 5.00V at 500K (227 deg C). 5V is equivalent to D1023 in 10-bits
; temperature (K) is A/D value (10-bits) x 500/1023. Take 273 to get deg C

; multiply by D500
MULT_500
	movf	TEMP_M,w
	movwf	AARGB0
	movf	TEMP_L,w
	movwf	AARGB1
	movlw	H'1'
	movwf	BARGB0
	movlw	H'F4'		; H1F4 for 500
	movwf	BARGB1
	call	FXM1616U	; multiply

; divide 
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
	movlw	H'3'
	movwf	BARGB0
	movlw	H'FF'		; D1023 = H3FF divisor
	movwf	BARGB1
	call	FXD2416U
; AARGB1 and AARGB2 has result of temperature in Kelvin
	movf	AARGB1,w
	movwf	KELVIN_0	; Kelvin temperature ms byte storage
	movf	AARGB2,w
	movwf	KELVIN_1	; Kelvin ls byte store

; check remainder
	btfss	REMB0,7		; if set increase value by 1
	goto	NO_INCREMENT1
	incf	KELVIN_1,f
	btfsc	STATUS,Z	; if zero increase ms byte
	incf	KELVIN_0,f
NO_INCREMENT1

; convert to -128 to +128 degrees (bit 7 set is -temperature)
; take 273 (H111) away
	movlw	H'1'		; ms byte subtraction
	subwf	KELVIN_0,w
	movwf	TEMPB0
	movlw	H'11'		; ls byte subtraction
	subwf	KELVIN_1,w	
	movwf	TEMPB1
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	TEMPB0,f	
	btfss	TEMPB0,7	; if set then negative result
	goto	SET_TEMP
; negative so reverse the subtraction
	movf	KELVIN_0,w	; ms byte subtraction
	sublw	H'1'
	movwf	TEMPB0
	movf	KELVIN_1,w	; ls byte subtraction
	sublw	H'11'	
	movwf	TEMPB1
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	TEMPB0,f
	bsf		TEMPB1,7	; set sign bit for negative
SET_TEMP
	movf	TEMPB1,w	; temperature from -127 to +127
	movwf	TEMP_C1		; temperature in degrees C, bit 7 is sign (- when set)

; second temperature calculation
; remeasure
	call	ACQUIRE_AD	; convert
	movf	ADRESH,w
	movwf	TEMPER_M	; temperature ms byte
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls bits
	bcf		STATUS,RP0	; bank 0
	movwf	TEMPER_L

; set for Channel 3
	bsf		ADCON0,3
	bsf		ADCON0,4	; channel 3

; temperature calibration 5.00V at 500K (227deg C). 5V is equivalent to D1F4 in 10-bits
; temperature (K) is A/D value (10-bits) x 500/1023. Take 273 to get deg C

; multiply by D500
MULT_500_2
	movf	TEMPER_M,w
	movwf	AARGB0
	movf	TEMPER_L,w
	movwf	AARGB1
	movlw	H'1'
	movwf	BARGB0
	movlw	H'F4'		; H1F4 for 500
	movwf	BARGB1
	call	FXM1616U	; multiply

; divide 
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
	movlw	H'3'
	movwf	BARGB0
	movlw	H'FF'		; D1023 = H3FF divisor
	movwf	BARGB1
	call	FXD2416U
; AARGB1 and AARGB2 has result of temperature in Kelvin
	movf	AARGB1,w
	movwf	KELVIN_0	; Kelvin temperature ms byte storage
	movf	AARGB2,w
	movwf	KELVIN_1	; Kelvin ls byte store

; check remainder
	btfss	REMB0,7		; if set increase value by 1
	goto	NO_INCREMENT2
	incf	KELVIN_1,f
	btfsc	STATUS,Z	; if zero increase ms byte
	incf	KELVIN_0,f
NO_INCREMENT2

; convert to -128 to +128 degrees (bit 7 set is - temperature)
; take 273 (H111) away
	movlw	H'1'		; ms byte subtraction
	subwf	KELVIN_0,w
	movwf	TEMPB0
	movlw	H'11'		; ls byte subtraction
	subwf	KELVIN_1,w	
	movwf	TEMPB1
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	TEMPB0,f	
	btfss	TEMPB0,7	; if set then negative result
	goto	SET_TEMP2
; negative so reverse the subtraction
	movf	KELVIN_0,w	; ms byte subtraction
	sublw	H'1'
	movwf	TEMPB0
	movf	KELVIN_1,w	; ls byte subtraction
	sublw	H'11'	
	movwf	TEMPB1
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	TEMPB0,f
	bsf		TEMPB1,7	; set sign bit for negative
SET_TEMP2
	movf	TEMPB1,w	; second calculation temperature from -127 to +127
	xorwf	TEMP_C1,w	; first calculation temperature in degrees C, bit 7 is sign (- when set)
	btfsc	STATUS,Z	; if the same 
	goto	USE_C		; use TEMP_C1 value if the same
; +/- 1 window allowed
	incf	TEMPB1,w
	xorwf	TEMP_C1,w
	btfsc	STATUS,Z
	goto	USE_C
	incf	TEMP_C1,w
	xorwf	TEMPB1,w
	btfss	STATUS,Z
	goto	SEND_DATA	; bypass readings because temperature does not match
USE_C
	movf	TEMP_C1,w	; use first calculation
	movwf	TEMP_C

; Turnover setting
; check if a cell voltage was a low value (less than 0.5V) if so alter turnover temperature compensation
; this is tied to 0V at power up to allow the flag to change TEMP_CAL flag set

	btfss	TEMP_CAL,0	; if set calibrate
	goto	CH_3AD		; no cal so bypass

; transfer values to cal
	movf	KELVIN_0,w	; ms byte of temperature
	movwf	TURNOVER_0
	movf	KELVIN_1,w
	movwf	TURNOVER_1
; write to EEPROM	
	movlw	EEPROM2		; turnover storage
	call	EEREAD		; sets EEADR
	movf	TURNOVER_0,w; ms byte
	call	EEWRITE	
	movlw	EEPROM3		; turnover storage
	call	EEREAD		; sets EEADR
	movf	TURNOVER_1,w; ls byte
	call	EEWRITE	

LED_SHOWx
	movlw	D'6'
	movwf	LED_PERIOD1	; counters for LED lit period
LOOP_Q
	bsf		PORTA,6		; green LED on
	btfss	TEMP_CAL,0
	bcf		PORTA,6		; green LED off if TEMP_CAL,0 clear
	movlw	D'255'
	movwf	LED_PERIOD2	; counter for LED lit period
LOOP_P
	call	DELAY		; on or off period
	decfsz	LED_PERIOD2,f; reduce to zero
	goto	LOOP_P
	decfsz	LED_PERIOD1,f; when zero end of LED display	
	goto	LOOP_R
	goto	BYPASS_LED
LOOP_R
	incf	TEMP_CAL,f	; toggle ls bit
	goto	LOOP_Q

CH_3AD ; pressure
	bsf		ADCON0,3
	bsf		ADCON0,4	; channel 3
	call	ACQUIRE_AD	; convert
	movf	ADRESH,w
	movwf	TEMP_M
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls bits
	bcf		STATUS,RP0	; bank 0
	movwf	TEMP_L
; reset to CH1
	bsf		ADCON0,3
	bcf		ADCON0,4	; channel 1

; calculate maximum span
; span is max value - min value or (3V - 1V = 2V at turnover temperature)
; span change with temperature is 2V (H199) x current temperature (273 plus current temperature in deg C)
; all divided by /turnover temperature 

; current temperature x 2V
; compensate according to COMPENSATE value (0 to 9)
; if compensate = 0 set temperature to turnover temperature so no compensation
	movf	COMPENSATE,w
	btfss	STATUS,Z
	goto	DO_COMPENSATE
  	movf	TURNOVER_0,w	; ms byte
	movwf	AARGB0
	movf	TURNOVER_1,w	; ls byte	
	movwf	AARGB1
	goto	MULT_SET

DO_COMPENSATE

	bcf		NEG,0		; clear sign bit

; take turnover temperature away
	movf	TURNOVER_0,w; ms byte subtraction
	subwf	KELVIN_0,w
	movwf	TEMPB0
	movf	TURNOVER_1,w; ls byte subtraction
	subwf	KELVIN_1,w	
	movwf	TEMPB1
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	TEMPB0,f	
	btfss	TEMPB0,7	; if set then negative result
	goto	BY_NEG
; negative so reverse the subtraction
	movf	KELVIN_0,w	; ms byte subtraction
	subwf	TURNOVER_0,w
	movwf	TEMPB0
	movf	KELVIN_1,w	; ls byte subtraction
	subwf	TURNOVER_1,w	
	movwf	TEMPB1
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	TEMPB0,f
	bsf		NEG,0		; set sign bit for negative
BY_NEG

; find proportion to compensate for temperature (9/9, 8/9,7/9...1/9) 

; multiply difference in temperature from turnover temperature by Compensate value
	movf	TEMPB0,w
	movwf	AARGB0		; ms byte clear
	movf	TEMPB1,w	; temperature offset from 0 deg C
	movwf	AARGB1
	clrf	BARGB0		; ms byte 0  
	movf	COMPENSATE,w; compensation value
	movwf	BARGB1		; ls byte set
	call	FXM1616U	; multiply

; divide by 9 for 11% steps from 11% to 100% of absolute change 
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
	clrf	BARGB0
	movlw	H'9'		; divisor
	movwf	BARGB1
	call	FXD2416U
; AARGB1 and AARGB2 has result 

; add or subtract result to turnover temperature
	btfsc	NEG,0		; when set negative so subtract
	goto	SUB_FROM_TURN
; add 	
	movf	TURNOVER_0,w
	addwf	AARGB1,f	; add to ms byte
	movf	TURNOVER_1,w
	addwf	AARGB2,f	; ls byte	
	btfsc	STATUS,C
	incf	AARGB1,f	; increase if ls overflow
	goto	SHIFT_FOR_MULT
SUB_FROM_TURN
; subtract
	movf	AARGB1,w	; ms byte subtraction
	subwf	TURNOVER_0,w;  ms byte
	movwf	AARGB1
	movf	AARGB2,w	; ls byte subtraction
	subwf	TURNOVER_1,w	
	movwf	AARGB2
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	AARGB1,f

SHIFT_FOR_MULT
	movf	AARGB1,w	; temperature
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1

; multiply by 2V
MULT_SET
	movlw	H'1'		; 2V = (1023 x 2/5) for A/D value or D409, H199 
	movwf	BARGB0
	movlw	H'99'		; H199 for D409 = 2V
	movwf	BARGB1
	call	FXM1616U	; multiply

; divide by turnover temperature
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
	movf	TURNOVER_0,w
	movwf	BARGB0
	movf	TURNOVER_1,w
	movwf	BARGB1
	call	FXD2416U
; AARGB1 and AARGB2 has result of span
	movf	AARGB1,w
	movwf	SPAN_H
	movf	AARGB2,w
	movwf	SPAN_L

; calculate maximum pressure value at 110%
;	calculate 110% span
	movf	SPAN_H,w	; span ms byte
	movwf	AARGB0
	movf	SPAN_L,w
	movwf	AARGB1
	movlw	H'0'		; 
	movwf	BARGB0
	movlw	D'110'		; x 110
	movwf	BARGB1
	call	FXM1616U	; multiply

; divide by 100
; shift multiply bytes left
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
	movlw	H'0'
	movwf	BARGB0
	movlw	D'100'		; 
	movwf	BARGB1
	call	FXD2416U
; AARGB1 and AARGB2 has result of span
	movf	AARGB1,w
	movwf	SPAN_H_110
	movf	AARGB2,w
	movwf	SPAN_L_110

; add 1V to 110% span 1V = D204 or HCC
	movf	SPAN_H_110,w	; ms byte of span
	movwf	MAX_PR_M	
	movlw	H'CC'			; 1V
	addwf	SPAN_L_110,w	; add
	movwf	MAX_PR_L
	btfsc	STATUS,C
	incf	MAX_PR_M,f	; increase if ls overflow

; subtract pressure value from maximum pressure value if negative set at 110%
	movf	TEMP_M,w	; pressure ms byte subtraction
	subwf	MAX_PR_M,w	;  ms byte
	movwf	TEMPB0
	movf	TEMP_L,w	; ls byte subtraction
	subwf	MAX_PR_L,w	
	movwf	TEMPB1
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	TEMPB0,f

	btfss	TEMPB0,7	; if negative 
	goto	SUB_MIN
	movlw	D'110'		; if negative set at 110%
	movwf	TANK_LEV	; set value to 110
	goto	SEND_DATA

; subtract minimum digital value D204 (HCC) from pressure value if negative set at 0%
SUB_MIN	
	movlw	H'00'		; ms byte subtraction
	subwf	TEMP_M,w	; pressure ms byte
	movwf	TEMPB0
	movlw	H'CC'		; ls byte subtraction
	subwf	TEMP_L,w	
	movwf	TEMPB1
	btfss	STATUS,C	; decrease ms byte if carry clear
	decf	TEMPB0,f

	btfss	TEMPB0,7	; if negative 
	goto	TRANSF_VAL
	clrf	TANK_LEV	; set value to 0
	goto	SEND_DATA

; convert to 0 to 100% level (pressure value after minimum digital value is subtacted x 100/span)
TRANSF_VAL	
; transfer values
	movf	TEMPB0,w
	movwf	AARGB0
	movf	TEMPB1,w
	movwf	AARGB1
	
	clrf	BARGB0
	movlw	D'100'		; x 100
	movwf	BARGB1
	call	FXM1616U	; multiply by 100
; shift bytes
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2
	movf	SPAN_H,w	; span ms byte divisor
	movwf	BARGB0
	movf	SPAN_L,w	; span ls byte
	movwf	BARGB1
	call	FXD2416U	; divide
;  AARGB2 has % result (max is 110%) to show overfull level
	movf	AARGB2,w
	movwf	TANK_LEV	; tank level %
	sublw	D'110'		; if 110
	btfsc	STATUS,C
	goto	SEND_DATA
	movlw	D'110'
	movwf	TANK_LEV

; Send Data

SEND_DATA

; start bits with gap for locking to data rate, encode value (ENCODE_VAL), tank number (TANK_No,)
; temperature deg C (TEMP_C), cell voltage % (CELL_PERC), tank level % (TANK_LEV)

; send out a 1 to engage the receiver

	bsf		PORTB,7		; send a 1
	movlw	D'200'		; 50 ms
	call	DELAYX		; ms
	bcf		PORTB,7		; send a zero
	movlw	D'20'		; 5 ms 
	call	DELAYX		; ms

; send a reference gap

	bsf		PORTB,7		; a 1
	movlw	D'64'		; 16ms 
	call	DELAYX		; extra length
	bcf		PORTB,7		; a 0		
	call	DELAY

; start sending data
; encode
; 4-bits
	swapf	ENCODE_VAL,w; encode (place at ms nibble)
; tank number
; 4-bits
	iorwf	TANK_No,w	; include tank number
	movwf	TEMP		; working register
	call	ROLL_8		; send out 8-bits
; tank level percentage (send twice to ensure value is correctly received)
; 8-bits
	movf	TANK_LEV,w
	movwf	TEMP		; working
	call	ROLL_8		; send out 8-bits
; temperature
; 8-bits
	movf	TEMP_C,w	; temperature
	movwf	TEMP		; working
	call	ROLL_8		; send out 8-bits
; cell voltage 
; 8-bits
	movf	CELL_VOLT,w	; cell voltage
	movwf	TEMP		; working
	call	ROLL_8		; send out 8-bits
; tank level percentage (second send)
; 8-bits
	movf	TANK_LEV,w
	movwf	TEMP		; working
	call	ROLL_8		; send out 8-bits
; temperature (second sending)
; 8-bits
	movf	TEMP_C,w	; temperature
	movwf	TEMP		; working
	call	ROLL_8		; send out 8-bits
; stop bits
	movlw	H'AA'		; send a value that is greater than any data so it is unique 
	movwf	TEMP
	call	ROLL_8
	bcf		PORTB,7		; output low
	goto	LEDS

; subroutines
; send out 8-bits
ROLL_8
	rlf		TEMP,f
	call	SET_OUT
	rlf		TEMP,f
	call	SET_OUT
	rlf		TEMP,f
	call	SET_OUT
	rlf		TEMP,f
	call	SET_OUT
	rlf		TEMP,f
	call	SET_OUT
	rlf		TEMP,f
	call	SET_OUT
	rlf		TEMP,f
	call	SET_OUT
	rlf		TEMP,f
	call	SET_OUT
	return
SET_OUT
	btfss	STATUS,C	; if set send a 1
	goto	CLR_7
	bsf		PORTB,7
	call	DELAY
	return
CLR_7	
	bcf		PORTB,7
	call	DELAY
	return

LEDS
; check if comparator wakeup. If so drive RGB LED
	btfss	COMP_FLG,0	; if comparator flag is set drive LEDs
	goto	BYPASS_LED
	
; drive RGB LED for a period before readying for sleep
DRV_LED
	movf	TANK_LEV,w
	sublw	D'90'
	btfss	STATUS,C	; if negative above 90% (White)
	goto	LED_90
	movf	TANK_LEV,w
	sublw	D'80'
	btfss	STATUS,C	; if negative above 80% (Violet)
	goto	LED_80
	movf	TANK_LEV,w
	sublw	D'70'
	btfss	STATUS,C	; if negative above 70% (Violet-Indigo)
	goto	LED_70
	movf	TANK_LEV,w
	sublw	D'60'
	btfss	STATUS,C	; if negative above 60% (Indigo)
	goto	LED_60
	movf	TANK_LEV,w
	sublw	D'50'
	btfss	STATUS,C	; if negative above 50% (Indigo-Blue)
	goto	LED_50
	movf	TANK_LEV,w
	sublw	D'40'
	btfss	STATUS,C	; if negative above 40% (blue)
	goto	LED_40
	movf	TANK_LEV,w
	sublw	D'30'
	btfss	STATUS,C	; if negative above 30% (Green)
	goto	LED_30
	movf	TANK_LEV,w
	sublw	D'20'
	btfss	STATUS,C	; if negative above 20% (Yellow)
	goto	LED_20
	movf	TANK_LEV,w
	sublw	D'10'
	btfss	STATUS,C	; if negative above 10% (Orange)
	goto	LED_10

LED_0 ; above 0 (and less than 10%) 
	clrf	MODULATE	; modulation of LED off < 10%(Red)
	bsf		PORTA,7		; red LED
	bcf		PORTA,6		; green off
	bcf		PORTA,0		; blue off
	goto	LED_CYCLE
LED_10 ; above 10%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,7		; red LED on
	bsf		MODULATE,6	; modulate green LED at 50%
	bcf		PORTA,0		; blue off
	goto	LED_CYCLE
LED_20 ; above 20%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,7		; red LED on
	bsf		PORTA,6		; green LED on
	bcf		PORTA,0		; blue off
	goto	LED_CYCLE
LED_30 ; above 30%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,6		; green LED on
	bcf		PORTA,0		; blue off
	bcf		PORTA,7		; red off
	goto	LED_CYCLE
LED_40 ; above 40%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bcf		PORTA,6		; green off
	bcf		PORTA,7		; red off
	goto	LED_CYCLE
LED_50 ; above 50%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		MODULATE,6	; modulate green LED at 50%	
	bcf		PORTA,7		; red off
	goto	LED_CYCLE
LED_60 ; above 60%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		PORTA,6		; green LED on
	bcf		PORTA,7		; red off
	goto	LED_CYCLE
LED_70 ; above 70%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		MODULATE,7	; modulate red LED at 50%
	bcf		PORTA,6		; green off
	goto	LED_CYCLE
LED_80 ; above 80%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		PORTA,7		; red LED on
	bcf		PORTA,6		; green off
	goto	LED_CYCLE
LED_90 ; above 90%
	clrf	MODULATE	; LED modulation off
	bsf		PORTA,0		; blue LED on
	bsf		PORTA,7		; red LED on
	bsf		PORTA,6		; green LED on

LED_CYCLE
	movlw	D'85'
	movwf	LED_PERIOD1	; counters for LED lit period
LOOP_U
	movlw	D'20'
	movwf	LED_PERIOD2	; counter for LED lit period
LOOP_B
	call	DELAY		; modulation period
; modulate LEDs
	btfsc	MODULATE,6	; if set switch green LED on and off
	call	MOD_GREEN
	btfsc	MODULATE,7
	call	MOD_RED
	decfsz	LED_PERIOD2,f; reduce to zero
	goto	LOOP_B
	decfsz	LED_PERIOD1,f; when zero end of LED display	
	goto	LOOP_U
	goto	BYPASS_LED	; end of LED display

MOD_GREEN
	btfss	PORTA,6		; if LED on switch off, if off switch on
	goto	GREEN_ON
	bcf		PORTA,6		; green LED off
	return
GREEN_ON
	bsf		PORTA,6		; green LED on
	return

MOD_RED
	btfss	PORTA,7		; if LED on switch off, if off switch  on
	goto	RED_ON
	bcf		PORTA,7		; red LED off
	return
RED_ON
	bsf		PORTA,7		; red LED on
	return

BYPASS_LED
; prepare for sleep
	bcf		PORTA,7		; red LED off
	bcf		PORTA,6		; green LED off
	bcf		PORTA,0		; blue LED off
	bsf		PORTA,4		; auxilliary power off
	movlw	D'20'		; 5ms to power down
	call	DELAYX

; wait till temperature input is low before sleep. 
; indicates that power is off

; temperature
RECH_TEMP
	bcf		ADCON0,3
	bsf		ADCON0,4	; channel 2
	call	ACQUIRE_AD	; convert
	movf	ADRESH,w
	btfss	STATUS,Z	; if ms byte not zero recheck	
	goto	RECH_TEMP	; recheck temperature
; ls bits
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls bits
	bcf		STATUS,RP0	; bank 0
; below 250?
	sublw	D'250'
	btfss	STATUS,C
	goto	RECH_TEMP

	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000101'	; comparator C2 on
	movwf	CMCON
	bcf		STATUS,RP0	; select memory bank 0
	bcf		ADCON0,ADON	; A/D off
	movlw	D'1'
	call	DELAYX		; 250us
	bcf		PIR2,CMIF	; clear after time to setup comparator interrupt flag

	clrwdt
	sleep
	nop
	clrwdt
	clrf	COMP_FLG	; comparator flag indicator initially cleared
	btfsc	PIR2,CMIF	; comparator interrupt flag if set then comparator interrupt
	bsf		COMP_FLG,0	; set comparator flag indicator if comparator flag is set
	bcf		PIR2,CMIF	; comparator flag cleared

; read AN2. If zero then a false comparator trigger. ie AN2 must be above 0V because S1 should have been 
; pressed to cause the comparator interrupt flag to be set.

	bsf		ADCON0,ADON	; A/D on
	
; measure temperature input
	bcf		ADCON0,3
	bsf		ADCON0,4	; channel 2
	movlw	D'20'		; 5ms to power up
	call	DELAYX
	call	ACQUIRE_AD	; convert
	movf	ADRESH,w
	movwf	TEMP_M		; temperature ms byte
	bsf		STATUS,RP0	; select memory bank 1
	movf	ADRESL,w	; ls bits
	bcf		STATUS,RP0	; bank 0
	movwf	TEMP_L

; check if a low value (less than 0.5V) if so clear comp flag

	movf	TEMP_M,w
	btfss	STATUS,Z	; if ms byte not zero bypass	
	goto	SET_INPUTS_OUTPUTS	; not a low value
; ls bits
	movf	TEMP_L,w	; ls bits
	sublw	D'102'		; below 102? (0.5V)
	btfsc	STATUS,C
	clrf	COMP_FLG	
	goto	SET_INPUTS_OUTPUTS


; ***************************************
; subroutine to wait for conversion

ACQUIRE_AD
; wait for >20us
	movlw	D'100'
	movwf	CONVERSION
LOOP_CONV
	decfsz	CONVERSION,f	; decrease 
	goto	LOOP_CONV	
	bsf		ADCON0,2		; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2		; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	return

; delay loop 

DELAY
	movlw	D'4'		; delay value
DELAYX
	movwf	STORE1		; STORE1 is number of loops value
LOOP8	
	movlw	D'165'
DELDSP
	movwf	STORE2		; STORE2 is internal loop value	
LOOP9
	decfsz	STORE2,f
	goto	LOOP9
	decfsz	STORE1,f
	goto	LOOP8		; decrease till STORE1 is zero
	return

; subroutine to read EEPROM memory 

EEREAD
	bcf		STATUS,RP0	; select bank 0
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM
EWRITE
EEWRITE	
	bcf 	STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 	STATUS,RP0	; select memory bank 3
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE	; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit

WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 	STATUS,RP0	; select memory bank 0
	return				; value written 


; 8 x 8 multiply

EIGHTEIGHT	    CLRF    AARGB1          ; clear partial product
UMUL0808L        
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
                MOVF    AARGB0,W

LOOPUM0808A
                RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    LUM0808NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808A

                CLRF    AARGB0
                RETLW   H'00'

LUM0808NAP
                BCF     STATUS,C
                GOTO    LUM0808NA

LOOPUM0808
                RRF     BARGB0, F
                BTFSC   STATUS,C
                ADDWF   AARGB0, F
LUM0808NA       RRF    	AARGB0, F
                RRF    	AARGB1, F
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM0808

                return             

; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U       	CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1,F
                MOVF            BARGB1,W
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                RLF             AARGB0,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LA 		RLF             AARGB0,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46L8         RLF             AARGB1,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LB         RLF             AARGB1,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46L16        RLF             AARGB2,F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1,F
                RLF             REMB0,F
                RLF             TEMPD,F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1,F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0,F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD,F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD,F
UOK46LC 		RLF             AARGB2,F
                DECFSZ          LOOPCOUNT,F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        	ADDWF           REMB1,F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0,F
UOK46L			RETURN

; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   0x08
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   0x00
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  0x08
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN


 end
